﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Xml.Serialization;
using MemoQ.Addins.Common.Utils;

namespace DummyTMPlugin
{
    /// <summary>
    /// Class for storing the Dummy TM plugin settings.
    /// </summary>
    /// <remarks>
    /// This class is forward compatible: an older version of the software should be able to serialize the
    /// class without deleting fields added in a newer version.
    /// </remarks>
    public class DummyTMSettings
    {
        /// <summary>
        /// Ensures forward compatibility.
        /// </summary>
        [XmlAnyElement]
        public System.Xml.XmlElement[] __forwardCompatibleElements;

        /// <summary>
        /// The path of the settings file.
        /// </summary>
        private static string settingsFilePath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData), @"MemoQ\Plugins\DummyTMSettings.xml");

        /// <summary>
        /// Indicates whether the plugin is enabled or not.
        /// </summary>
        public bool PluginEnabled = false;

        /// <summary>
        /// The user name used to be able to use the MT service.
        /// </summary>
        public string UserName = null;

        /// <summary>
        /// The encrypted password.
        /// </summary>
        public string Password = null;

        /// <summary>
        /// The decrypted password.
        /// </summary>
        [XmlIgnore]
        public string DecryptedPassword
        {
            get
            {
                if (Password == null) return null;
                else return DPAPIHelper.Decrypt(Password, DPAPIHelper.Store.User);
            }
            set
            {
                Password = DPAPIHelper.Encrypt(value, DPAPIHelper.Store.User);
            }
        }

        /// <summary>
        /// The token code used to be able to use the TM service.
        /// </summary>
        [XmlIgnore]
        public string TokenCode = null;

        /// <summary>
        /// The time when we have to refresh our token code.
        /// </summary>
        [XmlIgnore]
        public DateTime TokenCodeExpires = DateTime.MinValue;

        /// <summary>
        /// The languages supported by the TM service.
        /// </summary>
        public List<string> SupportedLanguages = null;

        /// <summary>
        /// Private constructor to avoid mulitple options instances.
        /// </summary>
        private DummyTMSettings()
        { }

        /// <summary>
        /// The singleton instance.
        /// </summary>
        private static DummyTMSettings instance;

        /// <summary>
        /// The singleton instance.
        /// </summary>
        public static DummyTMSettings Instance
        {
            get
            {
                if (instance == null)
                    instance = loadSettings();

                return instance;
            }
        }

        /// <summary>
        /// Loads the settings from the configuration if already
        /// exists, otherwise return the default settings.
        /// </summary>
        private static DummyTMSettings loadSettings()
        {
            DummyTMSettings settings = null;

            try
            {
                settings = SerializationHelper.DeserializeXMLFallbackToNullOnError<DummyTMSettings>(settingsFilePath);
            }
            catch (Exception)
            {
                settings = null;
            }
            finally
            {
                if (settings == null)
                    settings = new DummyTMSettings();
            }

            return settings;
        }

        /// <summary>
        /// Saves the setting to configuration file.
        /// </summary>
        public void SaveSettings()
        {
            if (!Directory.Exists(Path.GetDirectoryName(settingsFilePath)))
                Directory.CreateDirectory(Path.GetDirectoryName(settingsFilePath));

            SerializationHelper.SerializeXML(this, settingsFilePath);
        }

        /// <summary>
        /// Indicates whether the plugin is configured or not.
        /// </summary>
        public static bool PluginIsConfigured
        {
            get
            {   return File.Exists(settingsFilePath) &&
                    !string.IsNullOrEmpty(Instance.UserName) &&
                    !string.IsNullOrEmpty(Instance.Password) &&
                    Instance.SupportedLanguages != null &&
                    Instance.SupportedLanguages.Count > 0;
            }
        }
    }
}
