﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using MemoQ.Addins.Common.DataStructures;
using MemoQ.TBInterfaces;

namespace DummyTBPlugin
{
    class DummyTBService
    {
        private static DummyTBService instance;

        private Dictionary<string, string> tokenCodes;

        private DummyTBService()
        {
            tokenCodes = new Dictionary<string, string>();
        }

        public static DummyTBService Instance
        {
            get
            {
                if (instance == null)
                    instance = new DummyTBService();

                return instance;
            }
        }

        public string Login(string userName, string password)
        {
            // simulate network latency
            Thread.Sleep(3000);

            // the password has to be the user name twice
            if (password == userName)
            {
                if (!tokenCodes.ContainsKey(userName))
                    tokenCodes.Add(userName, Guid.NewGuid().ToString());

                return tokenCodes[userName];
            }
            else
                return null;
        }

        public List<string> ListSupportedLanguages(string tokenCode)
        {
            // simulate network latency
            Thread.Sleep(3000);

            if (tokenCodes.Values.Contains(tokenCode))
                return new List<string>() { "eng", "ger", "hun" };
            else
                throw new Exception("Invalid token code.");
        }

        public List<TerminologyResult> Lookup(Segment segment, string targetLangCode)
        {
            var random = new Random();
            var analyzedSegment = EnvironmentProvider.Environment.BuildWordsOfSegment(segment);

            // Maximum value for max is 3, now it is set to 1.
            var indexes = new int[Math.Min(new Random().Next(1, 4), analyzedSegment.Words.Length)];
            var results = new List<TerminologyResult>();

            // An empty segment or a segment only with tags.
            if (indexes.Length == 0)
                return results;

            indexes[0] = 0;
            if (indexes.Length > 1)
                indexes[1] = analyzedSegment.Words.Length - 1;
            if (indexes.Length > 2)
                indexes[2] = (int)(analyzedSegment.Words.Length / 2);

            foreach (int index in indexes)
            {
                var result = new TerminologyResult();
                var segmentBuilder = new SegmentBuilder();

                // Create segment with the specified text.
                foreach (var ch in segment.PlainText.Substring(analyzedSegment.Words[index].Start, analyzedSegment.Words[index].Length))
                    segmentBuilder.AppendChar(ch);
                var sourceTerm = segmentBuilder.ToSegment();
                result.ExternalId = Guid.NewGuid().ToString();
                result.SourceTerm = sourceTerm;
                result.StartPosInSegment = analyzedSegment.Words[index].Start;
                result.LengthInSegment = analyzedSegment.Words[index].Length;

                // The target language could come from the external source returning the terminology hit.
                // As we don't have that here, we have to manually set it based on what target language we
                // used for the search.
                result.TargetLanguage = targetLangCode;

                // Create Text with the spsicified text.
                segmentBuilder = new SegmentBuilder();
                string reversedText = reverseWord(sourceTerm.PlainText);
                foreach (var ch in reversedText)
                    segmentBuilder.AppendChar(ch);
                result.TargetTerm = segmentBuilder.ToSegment();

                result.Confidence = 0;

                result.EntryLevelMetadata = new List<Metadata>()
                {
                    new Metadata() { Name = "Entry-level meta 1", Value = "Entry-level meta value 1" },
                    new Metadata() { Name = "Entry-level meta 2", Value = "Entry-level meta value 2" }
                };
                result.LanguageLevelMetadataSource = new List<Metadata>()
                {
                    new Metadata() { Name = "Source lang-level meta 1", Value = "Source lang-level meta value 1" },
                    new Metadata() { Name = "Source lang-level meta 2", Value = "Source lang-level meta value 2" }
                };
                result.LanguageLevelMetadataTarget = new List<Metadata>()
                {
                    new Metadata() { Name = "Target lang-level meta 1", Value = "Target lang-level meta value 1" },
                    new Metadata() { Name = "Target lang-level meta 2", Value = "Target lang-level meta value 2" }
                };
                result.TermLevelMetadataSource = new List<Metadata>()
                {
                    new Metadata() { Name = "Source term-level meta 1", Value = "Source term-level meta value 1" },
                    new Metadata() { Name = "Source term-level meta 2", Value = "Source term-level meta value 2" }
                };
                result.TermLevelMetadataTarget = new List<Metadata>()
                {
                    new Metadata() { Name = "Target term-level meta 1", Value = "Target term-level meta value 1" },
                    new Metadata() { Name = "Target term-level meta 2", Value = "Target term-level meta value 2" }
                };

                // fill the pretty print html if the settings allows it
                if (DummyTBSettings.Instance.SendHtmlPreview)
                {
                    using (var sr = new StreamReader(Assembly.GetExecutingAssembly().GetManifestResourceStream("DummyTBPlugin.PrettyPrint.html")))
                        result.PrettyPrintHtml = sr.ReadToEnd();
                }

                result.Color = Color.FromArgb(random.Next(256), random.Next(256), random.Next(256));

                results.Add(result);
            }

            return results;
        }

        private static string reverseWord(string word)
        {
            var sb = new StringBuilder();
            for (int i = word.Length - 1; i >= 0; i--)
                sb.Append(word[i]);

            return sb.ToString();
        }
    }
}
