﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;

namespace DummyTBPlugin
{
    /// <summary>
    /// This class represents the options form of the dummy TB plugin.
    /// </summary>
    public partial class DummyTBOptionsForm : Form
    {
        private delegate void LoginDelegate(string userName, string password);

        private class LoginResult
        {
            public string UserName;
            public string Password;
            public bool LoginSuccessful;
            public List<string> SupportedLanguages;
            public Exception Exception;
        }

        private IAsyncResult asyncResult;

        public DummyTBOptionsForm()
        {
            InitializeComponent();

            // localization
            localizeContent();

            if (DummyTBSettings.PluginIsConfigured)
            {
                // the plugin is already configured
                tbUserName.Text = DummyTBSettings.Instance.UserName;
                tbPassword.Text = DummyTBSettings.Instance.DecryptedPassword;
                cbSendHtmlPreview.Checked = DummyTBSettings.Instance.SendHtmlPreview;
                foreach (string lang in DummyTBSettings.Instance.SupportedLanguages)
                {
                    lbLanguages.Items.Add(lang);
                }

                btnOK.Enabled = true;
            }
            else
            {
                // the plugin is not configured yet
                lnkRetrieveLangs.Enabled = btnOK.Enabled = false;
            }
        }

        private void localizeContent()
        {
            this.Text = LocalizationHelper.Instance.GetResourceString("OptionsFormCaption");
            this.lblUserName.Text = LocalizationHelper.Instance.GetResourceString("UserNameLabelText");
            this.lblPassword.Text = LocalizationHelper.Instance.GetResourceString("PasswordLabelText");
            this.lnkRetrieveLangs.Text = LocalizationHelper.Instance.GetResourceString("RetrieveLanguagesLinkText");
            this.lblSupportedLanguages.Text = LocalizationHelper.Instance.GetResourceString("SupportedLanguagesLabelText");
            this.cbSendHtmlPreview.Text = LocalizationHelper.Instance.GetResourceString("SendHtmlPreview");
            this.btnOK.Text = LocalizationHelper.Instance.GetResourceString("OkButtonText");
            this.btnCancel.Text = LocalizationHelper.Instance.GetResourceString("CancelButtonText");
        }

        private void tbUserNamePassword_TextChanged(object sender, EventArgs e)
        {
            lnkRetrieveLangs.Enabled = !string.IsNullOrEmpty(tbUserName.Text) && !string.IsNullOrEmpty(tbPassword.Text);
            btnOK.Enabled = false;
        }

        private void lnkRetrieveLangs_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
        {
            setControlsEnabledState(false);
            btnOK.Enabled = false;
            lbLanguages.Items.Clear();

            // do the update in the background
            asyncResult = new LoginDelegate(loginCore).BeginInvoke(
                tbUserName.Text, tbPassword.Text, new AsyncCallback(handleLoginFinished),
                new LoginResult() { UserName = tbUserName.Text, Password = tbPassword.Text });
        }

        private void loginCore(string userName, string password)
        {
            try
            {
                // try to login
                string tokenCode = DummyTBService.Instance.Login(userName, password);

                if (string.IsNullOrEmpty(tokenCode))
                {
                    // invalid user name or password
                    (asyncResult.AsyncState as LoginResult).LoginSuccessful = false;
                }
                else
                {
                    // successful login
                    (asyncResult.AsyncState as LoginResult).LoginSuccessful = true;
                    // try to get the list of the supported languages
                    (asyncResult.AsyncState as LoginResult).SupportedLanguages = DummyTBService.Instance.ListSupportedLanguages(tokenCode);
                }
            }
            catch (Exception ex)
            {
                (asyncResult.AsyncState as LoginResult).Exception = ex;
            }
        }

        private void handleLoginFinished(IAsyncResult result)
        {
            // it is possible that the form has disposed during the background operation (e.g. the user clicked on the cancel button)
            if (!IsDisposed)
            {
                Invoke(new MethodInvoker(delegate()
                    {
                        LoginResult loginResult = asyncResult.AsyncState as LoginResult;

                        if (loginResult.Exception != null)
                        {
                            // there was an error, display for the user
                            string caption = LocalizationHelper.Instance.GetResourceString("CommunicationErrorCaption");
                            string text = LocalizationHelper.Instance.GetResourceString("CommunicationErrorText");
                            MessageBox.Show(this, string.Format(text, loginResult.Exception.Message), caption, MessageBoxButtons.OK, MessageBoxIcon.Error);
                        }
                        else if (!loginResult.LoginSuccessful)
                        {
                            // the user name of the password is invalid, display for the user
                            string caption = LocalizationHelper.Instance.GetResourceString("InvalidUserNameCaption");
                            string text = LocalizationHelper.Instance.GetResourceString("InvalidUserNameText");
                            MessageBox.Show(this, text, caption, MessageBoxButtons.OK, MessageBoxIcon.Error);
                        }
                        else
                        {
                            // we have managed to get the supported languages, display them in the listbox
                            lbLanguages.Items.Clear();
                            foreach (string lang in loginResult.SupportedLanguages)
                            {
                                lbLanguages.Items.Add(lang);
                            }

                            btnOK.Enabled = loginResult.SupportedLanguages.Count > 0;
                        }

                        setControlsEnabledState(true);
                    }));
            }
        }

        private void setControlsEnabledState(bool enabled)
        {
            tbUserName.Enabled = enabled;
            tbPassword.Enabled = enabled;
            lnkRetrieveLangs.Enabled = enabled;
            cbSendHtmlPreview.Enabled = enabled;
            progressBar.Visible = !enabled;
        }

        private void DummyMTOptionsForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == System.Windows.Forms.DialogResult.OK)
            {
                // if there was a modification, we have to save the changes
                bool save = false;
                if (asyncResult != null)
                {
                    LoginResult loginResult = asyncResult.AsyncState as LoginResult;
                    DummyTBSettings.Instance.UserName = loginResult.UserName;
                    DummyTBSettings.Instance.DecryptedPassword = loginResult.Password;
                    DummyTBSettings.Instance.SupportedLanguages = loginResult.SupportedLanguages;
                    save = true;
                }
                if (cbSendHtmlPreview.Checked != DummyTBSettings.Instance.SendHtmlPreview)
                {
                    DummyTBSettings.Instance.SendHtmlPreview = cbSendHtmlPreview.Checked;
                    save = true;
                }

                if (save)
                    DummyTBSettings.Instance.SaveSettings();
            }
        }
    }
}
