﻿using System;
using System.Collections.Generic;
using System.IO;
using MemoQ.Addins.Common.Utils;

namespace DummyTBPlugin
{
    /// <summary>
    /// Class for storing the Dummy TB plugin settings.
    /// </summary>
    [Serializable]
    public class DummyTBSettings
    {
        /// <summary>
        /// The path of the settings file.
        /// </summary>
        private static string settingsFilePath;

        static DummyTBSettings()
        {
            settingsFilePath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData), "MemoQ", "Plugins", "DummyTBSettings.xml");
        }

        /// <summary>
        /// Indicates whether the plugin is enabled or not.
        /// </summary>
        public bool PluginEnabled = false;

        /// <summary>
        /// The user name used to be able to use the MT service.
        /// </summary>
        public string UserName = null;

        /// <summary>
        /// The encrypted password.
        /// </summary>
        public string Password = null;

        /// <summary>
        /// The decrypted password.
        /// </summary>
        public string DecryptedPassword
        {
            get
            {
                if (Password == null) return null;
                else return DPAPIHelper.Decrypt(Password, DPAPIHelper.Store.User);
            }
            set
            {
                Password = DPAPIHelper.Encrypt(value, DPAPIHelper.Store.User);
            }
        }

        /// <summary>
        /// The languages supported by the service.
        /// </summary>
        public List<string> SupportedLanguages = null;

        /// <summary>
        /// Indicates whether the plugin sends the html preview provided by the service.
        /// </summary>
        public bool SendHtmlPreview = false;

        /// <summary>
        /// Private constructor to avoid mulitple options instances.
        /// </summary>
        private DummyTBSettings()
        { }

        /// <summary>
        /// The singleton instance.
        /// </summary>
        [NonSerialized]
        private static DummyTBSettings instance;

        /// <summary>
        /// The singleton instance.
        /// </summary>
        public static DummyTBSettings Instance
        {
            get
            {
                if (instance == null)
                    instance = loadSettings();

                return instance;
            }
        }

        /// <summary>
        /// Loads the settings from the configuration if already
        /// exists, otherwise return the default settings.
        /// </summary>
        private static DummyTBSettings loadSettings()
        {
            var settings = SerializationHelper.DeserializeXMLFallbackToNullOnError<DummyTBSettings>(settingsFilePath);
            if (settings == null)
                settings = new DummyTBSettings();

            return settings;
        }

        /// <summary>
        /// Saves the setting to configuration file.
        /// </summary>
        public void SaveSettings()
        {
            if (!Directory.Exists(Path.GetDirectoryName(settingsFilePath)))
                Directory.CreateDirectory(Path.GetDirectoryName(settingsFilePath));

            SerializationHelper.SerializeXML(this, settingsFilePath);
        }

        /// <summary>
        /// Indicates whether the plugin is configured or not.
        /// </summary>
        /// <returns></returns>
        public static bool PluginIsConfigured
        {
            get
            {
                return File.Exists(settingsFilePath) &&
                    !string.IsNullOrEmpty(Instance.UserName) &&
                    !string.IsNullOrEmpty(Instance.Password) &&
                    Instance.SupportedLanguages != null &&
                    Instance.SupportedLanguages.Count > 0;
            }
        }
    }
}
