﻿using System.Drawing;
using System.Reflection;
using System.Windows.Forms;
using MemoQ.Addins.Common.Framework;
using MemoQ.TMInterfaces;

namespace DummyTMPlugin
{
    /// <summary>
    /// The main class of the Dummy TM plugin.
    /// </summary>
    public class DummyTMPluginDirector : PluginDirectorBase
    {
        /// <summary>
        /// The identifier of the plugin.
        /// </summary>
        public const string PluginId = "DummyTM";

        /// <summary>
        /// Cleans up the resources used by the plugin if needed.
        /// </summary>
        public override void Cleanup()
        {
            // write your cleanup code here
        }

        /// <summary>
        /// Gets the copyright text to show in memoQ's Tools / Options.
        /// </summary>
        public override string CopyrightText
        {
            get { return "(C) Kilgray"; }
        }

        /// <summary>
        /// Create an TM engine for the supplied language pair.
        /// </summary>
        public override IEngine CreateEngine(string srcLangName, string trgLangName)
        {
            return new DummyTMEngine(srcLangName, trgLangName);
        }

        /// <summary>
        /// Gets a 48x48 display icon to show in memoQ's Tools / Options. Black is the transparent color.
        /// </summary>
        public override Image DisplayIcon
        {
            get { return Image.FromStream(Assembly.GetExecutingAssembly().GetManifestResourceStream("DummyTMPlugin.Icon.bmp")); }
        }

        /// <summary>
        /// Gets the friendly name to show in memoQ's Tools / Options.
        /// </summary>
        public override string FriendlyName
        {
            get { return "Dummy TM Plugin"; }
        }

        /// <summary>
        /// Initializes the plugin.
        /// </summary>
        /// <param name="env"></param>
        public override void Initialize(IModuleEnvironment env)
        {
            // write your initialization code here
        }

        /// <summary>
        /// Gets whether the plugin is activated.
        /// </summary>
        public override bool IsActivated
        {
            get { return true; }
        }

        /// <summary>
        /// Gets if the plugin supports the provided language combination. The strings provided are memoQ languages codes.
        /// </summary>
        public override bool IsLanguagePairSupported(string srcLangName, string trgLangName)
        {
            return DummyTMSettings.Instance.SupportedLanguages != null &&
                DummyTMSettings.Instance.SupportedLanguages.Contains(srcLangName) &&
                DummyTMSettings.Instance.SupportedLanguages.Contains(trgLangName);
        }

        /// <summary>
        /// Gets whether the plugin is configured.
        /// </summary>
        public override bool PluginConfigured
        {
            get { return DummyTMSettings.PluginIsConfigured; }
        }

        /// <summary>
        /// Gets or sets whether the plugin is enabled.
        /// </summary>
        public override bool PluginEnabled
        {
            get
            {
                return DummyTMSettings.Instance.PluginEnabled;
            }
            set
            {
                if (DummyTMSettings.Instance.PluginEnabled != value)
                {
                    DummyTMSettings.Instance.PluginEnabled = value;
                    DummyTMSettings.Instance.SaveSettings();

                    environment.PluginAvailabilityChanged();
                }
            }
        }

        /// <summary>
        /// Gets the plugin's non-localized name.
        /// </summary>
        public override string PluginID
        {
            get { return "DummyTM"; }
        }

        /// <summary>
        /// Shows the plugin's options/about form
        /// </summary>
        public override void ShowOptionsForm(Form parentForm)
        {
            using (DummyTMOptionsForm form = new DummyTMOptionsForm())
            {
                if (form.ShowDialog(parentForm) == DialogResult.OK)
                {
                    environment.PluginAvailabilityChanged();
                }
            }
        }

        /// <summary>
        /// Indicates whether the plugin supports the segment lookup.
        /// </summary>
        public override bool SupportsSegmentLookup
        {
            get { return true; }
        }

        /// <summary>
        /// Indicates whether the plugin supports concordance lookup.
        /// </summary>
        public override bool SupportsConcordance
        {
            get { return true; }
        }

        public override IEnvironment Environment
        {
            set
            {
                this.environment = value;

                // initialize the localization helper
                LocalizationHelper.Instance.SetEnvironment(value);
            }
        }
    }
}
