﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using DummyTMInterface;

namespace DummyTMPlugin
{
    /// <summary>
    /// Helper class to be able to communicate with the web service.
    /// </summary>
    internal class DummyTMServiceHelper
    {
        public static ITMService getNewProxy()
        {
            var epAddr = new EndpointAddress("net.tcp://localhost:8733/TMService/DummyService");
            return ChannelFactory<ITMService>.CreateChannel(new NetTcpBinding(), epAddr);
        }

        /// <summary>
        /// Gets the valid token code.
        /// </summary>
        /// <returns>The token code.</returns>
        public static string GetTokenCode()
        {
            if (DummyTMSettings.Instance.TokenCodeExpires.Equals(DateTime.MinValue) || DummyTMSettings.Instance.TokenCodeExpires < DateTime.Now)
            {
                // refresh the token code
                var proxy = getNewProxy();
                using (proxy as IDisposable)
                {
                    DummyTMSettings.Instance.TokenCode = proxy.Login(DummyTMSettings.Instance.UserName, DummyTMSettings.Instance.DecryptedPassword);
                    DummyTMSettings.Instance.TokenCodeExpires = DateTime.Now.AddMinutes(1);
                    DummyTMSettings.Instance.SaveSettings();
                }
            }

            return DummyTMSettings.Instance.TokenCode;
        }

        /// <summary>
        /// Calls the web service's login method.
        /// </summary>
        /// <param name="userName">The user name.</param>
        /// <param name="password">The password.</param>
        /// <returns>The token code.</returns>
        public static string Login(string userName, string password)
        {
            var proxy = getNewProxy();
            using (proxy as IDisposable)
            {
                return proxy.Login(userName, password);
            }
        }

        /// <summary>
        /// Lists the supported languages of the dummy TM service.
        /// </summary>
        /// <param name="tokenCode">The token code.</param>
        /// <returns>The list of the supported languages.</returns>
        public static List<string> ListSupportedLanguages(string tokenCode)
        {
            var proxy = getNewProxy();
            using (proxy as IDisposable)
            {
                string[] supportedLanguages = proxy.ListSupportedLanguages(tokenCode).ToArray();

                if (supportedLanguages == null)
                {
                    // token expired
                    supportedLanguages = proxy.ListSupportedLanguages(GetTokenCode()).ToArray();
                }

                return supportedLanguages.ToList();
            }
        }

        /// <summary>
        /// Gets the possible translations for a single string from the translation memory.
        /// </summary>
        /// <param name="tokenCode">The token code.</param>
        /// <param name="input">The string to translate.</param>
        /// <param name="srcLangCode">The source language code.</param>
        /// <param name="trgLangCode">The target language code.</param>
        /// <returns>The possible translations of the input string.</returns>
        public static TMServiceLookupHit[] Lookup(string tokenCode, string input, string srcLangCode, string trgLangCode)
        {
            var proxy = getNewProxy();
            using (proxy as IDisposable)
            {
                TMServiceLookupHit[] results = proxy.Lookup(tokenCode, input, srcLangCode, trgLangCode);

                if (results == null)
                {
                    // token expired
                    results = proxy.Lookup(tokenCode, input, srcLangCode, trgLangCode);
                }

                return results;
            }
        }

        /// <summary>
        /// Gets the possible concordance lookup results of specific texts.
        /// </summary>
        /// <param name="tokenCode">The token code.</param>
        /// <param name="texts">The texts sent for concordance.</param>
        /// <param name="srcLangCode">The source language code.</param>
        /// <param name="trgLangCode">The target language code.</param>
        /// <param name="caseSensitive">The case sensitivity.</param>
        /// <param name="numericEquivalence">Numeric equivalence.</param>
        /// <param name="limit">The maximum number of total concordance results.</param>
        /// <param name="totalResultCount">The total number of concordance results.</param>
        /// <returns>The concordance results of the texts.</returns>
        public static TMServiceConcordanceItem[] Concordance(string tokenCode, string[] texts, string srcLangCode, string trgLangCode, bool caseSensitive, bool numericEquivalence, int limit, out int totalResultCount)
        {
            var proxy = getNewProxy();
            using (proxy as IDisposable)
            {
                TMServiceConcordanceItem[] results = proxy.Concordance(tokenCode, texts, srcLangCode, trgLangCode, caseSensitive, numericEquivalence, limit, out totalResultCount);

                if (results == null)
                {
                    // token expired
                    results = proxy.Concordance(tokenCode, texts, srcLangCode, trgLangCode, caseSensitive, numericEquivalence, limit, out totalResultCount);
                }

                return results;
            }
        }
    }
}
