﻿using System;
using System.Collections.Generic;
using System.Threading;
using DummyTMInterface;

namespace DummyTMService
{
    /// <summary>
    /// Dummy web service which can be called by the dummy TM plugin.
    /// </summary>
    public class TMService : ITMService
    {
        public TMService()
        {
            //Uncomment the following line if using designed components
            //InitializeComponent();
        }

        /// <summary>
        /// Call this method to get a token code for the further
        /// calls.Returns the token code if the credentials are
        /// valid (this dummy service allows the login request
        /// if the user name and the password are identical), 
        /// otherwise returns null.
        /// </summary>
        /// <param name="userName">The user name.</param>
        /// <param name="password">The password.</param>
        /// <returns>The token code if the credentials are valid, null otherwise.</returns>
        public string Login(string userName, string password)
        {
            // simulate the network latency
            Thread.Sleep(3000);

            return userName.Equals(password) ? TokenCodeGenerator.Instance.GenerateTokenCode(userName) : null;
        }

        /// <summary>
        /// Call this method to get the supported languages of the service.
        /// </summary>
        /// <param name="tokenCode">The token code.</param>
        /// <returns>The supported languages.</returns>
        public List<string> ListSupportedLanguages(string tokenCode)
        {
            // simulate the network latency
            Thread.Sleep(3000);

            if (!TokenCodeGenerator.Instance.TokenCodeIsValid(tokenCode))
                return null;

            return new List<string>() { "eng", "fre", "ger", "ita", "por" };
        }

        /// <summary>
        /// Call this method to get the possible translations for a single string.
        /// </summary>
        /// <param name="tokenCode">The token code.</param>
        /// <param name="input">The input string.</param>
        /// <param name="srcLangCode">The code of the source language.</param>
        /// <param name="trgLangCode">The code of the target language.</param>
        /// <returns>The possible translations of the input string.</returns>
        public TMServiceLookupHit[] Lookup(string tokenCode, string input, string srcLangCode, string trgLangCode)
        {
            // simulate the network latency
            Thread.Sleep(1000);

            if (!TokenCodeGenerator.Instance.TokenCodeIsValid(tokenCode))
                return null;

            // creating dummy hits with different match rates
            List<TMServiceLookupHit> results = new List<TMServiceLookupHit>();
            for (int i = 1; i <= 5; ++i)
                results.Add(new TMServiceLookupHit
                {
                    SourceString = input,
                    TargetString = string.Format("{0} - #{1}", input, i),
                    MatchRate = i * 19,
                    LastModified = DateTime.Now,
                    UserName = "DummyUser",
                    CreationDate = DateTime.Now,
                    CreatingUser = "DummyUser",
                    Subject = "DummySubject",
                    Domain = "DummyDomain",
                    Document = "DummyDocument"
                });

            return results.ToArray();
        }

        /// <summary>
        /// Gets the possible concordance lookup results of specific texts.
        /// </summary>
        /// <param name="tokenCode">The token code.</param>
        /// <param name="texts">The texts sent for concordance.</param>
        /// <param name="srcLangCode">The source language code.</param>
        /// <param name="trgLangCode">The target language code.</param>
        /// <param name="caseSensitive">The case sensitivity.</param>
        /// <param name="numericEquivalence">Numeric equivalence.</param>
        /// <param name="limit">The maximum number of total concordance results.</param>
        /// <param name="totalResultCount">The total number of concordance results.</param>
        /// <returns>The concordance results of the texts.</returns>
        public TMServiceConcordanceItem[] Concordance(string tokenCode, string[] texts, string srcLangCode, string trgLangCode, bool caseSensitive, bool numericEquivalence, int limit, out int totalResultCount)
        {
            // simulate the network latency
            Thread.Sleep(1000);

            if (!TokenCodeGenerator.Instance.TokenCodeIsValid(tokenCode))
            {
                totalResultCount = 0;
                return null;
            }

            int itemCount = limit < 5 ? limit : 5;

            string cleanText = string.Join(" ", texts).Replace("*", "");

            // Creating dummy concordance items
            List<TMServiceConcordanceItem> results = new List<TMServiceConcordanceItem>();
            for (int i = 1; i <= itemCount; ++i)
            {
                results.Add(new TMServiceConcordanceItem
                {
                    SourceString = string.Format("{0}prefix - {2} {1} {0}suffix - {2}", srcLangCode, cleanText, i),
                    TargetString = string.Format("{0}prefix - {2} {1} {0}suffix - {2}", trgLangCode, cleanText, i),
                    LastModified = DateTime.Now,
                    UserName = "DummyUser",
                    CreationDate = DateTime.Now,
                    CreatingUser = "DummyUser",
                    Subject = "DummySubject",
                    Domain = "DummyDomain",
                    Document = "DummyDocument",
                    RangeInfos = new TMServiceConcordanceItem.TMServiceConcordanceRangeInfo[]
                    {
                        new TMServiceConcordanceItem.TMServiceConcordanceRangeInfo
                        {
                            Start = (string.Format("{0}prefix - {1} ", srcLangCode, i)).Length,
                            Length = cleanText.Length
                        }
                    }
                });
            }

            totalResultCount = 5;
            return results.ToArray();
        }
    }
}
